import React, { useState, useEffect, useRef } from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { useTheme, ThemeVariant } from '@/context/ThemeContext';
import { useAuth } from '@/context/AuthContext';
import './LandingPage.css';

interface ThemeButton {
  name: string;
  path: string;
  color: string;
  textColor?: string;
  variant: ThemeVariant;
}

const themeButtons: ThemeButton[] = [
  { name: 'Simple', path: '/simple', color: '#ffffff', variant: 'simple' },
  { name: 'Floral', path: '/floral', color: '#a58ab5', variant: 'floral' },
  { name: 'Sunset Dream', path: '/nightsky', color: '#f06e9e', variant: 'nightSky' },
  { name: 'Cabin', path: '/cabin', color: '#423230', textColor: '#ffffff', variant: 'cabin' }
];

// Forest background images
const forestImages = [
  '/images/Elements/forestclouds1.jpg',
  '/images/Elements/forestclouds2.jpg',
  '/images/Elements/forestclouds3.jpg',
  '/images/Elements/forestclouds4.jpg',
  '/images/Elements/forestclouds5.jpg',
  '/images/Elements/forestclouds6.jpg',
  '/images/Elements/forestclouds7.jpg',
  '/images/Elements/forestclouds8.jpg',
];

const LandingPage: React.FC = () => {
  const [isPlaying, setIsPlaying] = useState(false);
  const [showPaths, setShowPaths] = useState(false);
  const [typedText, setTypedText] = useState('');
  const [showFinalText, setShowFinalText] = useState(false);
  const [backgroundImage, setBackgroundImage] = useState('');
  const navigate = useNavigate();
  const audioRef = useRef<HTMLAudioElement | null>(null);
  const { currentMode, setCurrentVariant, toggleMode } = useTheme();
  const { currentUser, logout } = useAuth();

  // Select a random forest image on component mount
  useEffect(() => {
    const randomIndex = Math.floor(Math.random() * forestImages.length);
    setBackgroundImage(forestImages[randomIndex]);
  }, []);

  const typeText = async () => {
    const text = "You reach a path in the woods.";
    let currentText = '';
    
    for (let char of text) {
      currentText += char;
      setTypedText(currentText);
      // Uncomment when audio file is provided
      // if (audioRef.current) {
      //   audioRef.current.currentTime = 0;
      //   audioRef.current.play();
      // }
      await new Promise(resolve => setTimeout(resolve, 100)); // Adjust typing speed
    }
    
    setTimeout(() => {
      setShowPaths(true);
      setTimeout(() => setShowFinalText(true), 1500);
    }, 500);
  };

  const handlePlay = () => {
    setIsPlaying(true);
    typeText();
  };

  const handleThemeClick = (variant: ThemeVariant) => {
    setCurrentVariant(variant); // Set the theme in context
    navigate('/home'); // Navigate to /home
  };

  const renderDoorways = () => {
    // ... existing code ...
  };

  return (
    <div className="landing-page" style={{ backgroundImage: `url(${backgroundImage})` }}>
      {/* Container for top-right controls */}
      <div className="top-right-controls-container">
        {/* Authentication Controls */}
        <div className="auth-controls-landing">
          {currentUser ? (
            <button onClick={async () => {
              try {
                await logout();
                // Optional: navigate to home or login page after logout
                // navigate('/'); 
              } catch (error) {
                console.error("Failed to log out", error);
              }
            }} className="auth-button logout-button">
              Logout ({currentUser.email?.substring(0, currentUser.email.indexOf('@')) || 'User'})
            </button>
          ) : (
            <>
              <Link to="/login" className="auth-button login-button">Login</Link>
              <Link to="/signup" className="auth-button signup-button">Sign Up</Link>
            </>
          )}
        </div>

        {/* Mode toggle button - small version */}
        <div className="landing-mode-toggle">
          <button 
            onClick={toggleMode}
            className={`toggle-switch ${currentMode}`}
            aria-label={`Switch to ${currentMode === 'light' ? 'dusky' : 'light'} mode`}
          >
            <span className="toggle-slider"></span>
            <span className="toggle-icon">{currentMode === 'light' ? '☀️' : '🌙'}</span>
          </button>
        </div>
      </div>
      
      {/* Center column without forest */}
      <div className="center-column"></div>
      
      {/* Audio element for typewriter sound */}
      <audio ref={audioRef}>
        <source src="/path/to/typewriter-sound.mp3" type="audio/mpeg" />
      </audio>

      {/* Initial state with play button */}
      {!isPlaying && (
        <div className="initial-state">
          <div className="play-circle">
            <button className="play-button" onClick={handlePlay}>
              <div className="play-icon"></div>
            </button>
          </div>
          <div className="theme-buttons">
            {themeButtons.map((theme) => (
              <button
                key={theme.name}
                className="theme-button"
                style={{
                  backgroundColor: theme.color,
                  color: theme.textColor || '#000000'
                }}
                onClick={() => handleThemeClick(theme.variant)}
              >
                {theme.name}
              </button>
            ))}
          </div>
          
          <div className="wide-mode-toggle-slider" onClick={toggleMode}>
            <div className={`slider-option light-option ${currentMode === 'light' ? 'active' : ''}`}>
              Light Mode
            </div>
            <div className={`slider-option dusky-option ${currentMode === 'dusky' ? 'active' : ''}`}>
              Dark Mode
            </div>
          </div>
        </div>
      )}

      {/* Animated text and paths */}
      {isPlaying && (
        <div className="animated-content">
          <div className="typed-text">
            {typedText}<span className="cursor">|</span>
          </div>
          
          {showPaths && (
            <>
              {showFinalText && (
                <div className="final-text">where does it go?</div>
              )}
              <div className="paths-container">
                {themeButtons.map((theme, index) => (
                  <div key={theme.name} className="path-group">
                    <div className={`path path-${index + 1}`}></div>
                    <div 
                      className="doorway"
                      style={{ backgroundColor: theme.color }}
                      onClick={() => handleThemeClick(theme.variant)}
                    >
                      <span style={{ color: theme.textColor || '#000000' }}>
                        {theme.name}
                      </span>
                    </div>
                  </div>
                ))}
              </div>
              
              <div className="wide-mode-toggle-slider paths-toggle" onClick={toggleMode}>
                <div className={`slider-option light-option ${currentMode === 'light' ? 'active' : ''}`}>
                  Light Mode
                </div>
                <div className={`slider-option dusky-option ${currentMode === 'dusky' ? 'active' : ''}`}>
                  Dark Mode
                </div>
              </div>
            </>
          )}
        </div>
      )}
    </div>
  );
};

export default LandingPage;
 